#!/usr/bin/perl

# Dieses Plugin kann mit ePN ausgefuehrt werden.
# nagios: +epn

use strict;
use warnings;

use Getopt::Long;
use Filesys::Df;

sub main(){
	# Befehlszeile parsen, Warn-Schwelle in $warn und kritische Schwelle in
	# $crit speichern.
	new Getopt::Long::Parser(config => ['no_ignore_case', 'bundling'])
  	->getoptions('warning|w=s' => \my $warn, 'critical|c=s' => \my $crit,
  	             'dir|d=s'     => \my $dir)
	  	|| do{
					print("UNKNOWN : Unable to parse arguments!\n");
					exit(3);
				};
	
	# $dir muss definiert...
	unless(defined($dir) && $dir ne ''){
		print("UNKNOWN: No directory given to monitor.\n");
		exit(3);
	}
	# ... und ein existierendes Verzeichnis sein.
	unless(-d $dir){
		printf("UNKNOWN: Directory not found: %s\n", $dir);
		exit(3);
	}
	
	# Default-Exitcode.
	my $status = 0;
	
	# Wir interessieren uns fuer die Angabe in MiB.
	my $df = df($dir, 1024*1024);
	
	# Speichere bool'sche Werte, ob eine der Schwellen verletzt ist
	# (oder undef, falls Parsing fehlschlaegt).
	my $isCrit = isOutOfRange($df->{per}, $crit);
	my $isWarn = isOutOfRange($df->{per}, $warn);
	
	# Werte Schwellwertverletzungen aus.
	if(!defined($isCrit) || !defined($isWarn)){
		print("UNKNOWN: Bad threshold format");
		$status = 3;
	}
	elsif($isCrit){
		printf("CRITICAL: Disk %s usage %d%% (out of threshold %s)", $dir, $df->{per}, $crit);
		$status = 2;
	}
	elsif($isWarn){
		printf("WARNING: Disk %s usage %d%% (out of threshold %s)", $dir, $df->{per}, $warn);
		$status = 1;
	}
	else{
		printf("OK: Disk %s usage %d%%", $dir, $df->{per});
	}
	
	# Gebe Performance-Daten zum belegten Speicherplatz sowie, sofern verfuegbar, den
	# belegten Inodes aus.
	printf(" | 'Space Usage'=%dMB;;;0;%d", $df->{used}, $df->{blocks});
	if(exists($df->{files})){
		printf(" 'Inode Usage'=%d;;;0;%d", $df->{fused}, $df->{files});
	}
	
	# Schliesse die Zeile mit Newline ab und beende mit dem oben festgelegten Exitcode.
	print("\n");
	exit($status);
}

# Pruefe, ob $value ausserhalb von $range liegt.
sub isOutOfRange($$){
	my($value, $range) = @_;
	
	# Wenn keine Schwelle definiert ist, liegt der Wert nicht ausserhalb.
	return(0) unless(defined($range));
	
	# Die Schwelle parsen und undef zurueckgeben, falls das Parsen fehlschlaegt.
	my($type, $min, $max) = parseThreshold($range);
	return(undef) unless(defined($type));
	
	# $oor erhaelt einen wahren bool'schen Wert, falls der Mindest- oder
	# Hoechstwert der Schwelle, sofern definiert, unter-/ueberschritten ist.
	my $oor = ((defined($min) && $value < $min) || (defined($max) && $value > $max));
	
	# $oor zurueckgeben, falls die Schwelle vom Typ "Alarm falls innerhalb" ist,
	# negiert.
	return($type ? !$oor : $oor);
}

# Zerlege eine Threshold-Range in Typ (innerhalb/ausserhalb des Bereichs) sowie
# untere und obere Schwelle, sofern definiert.
sub parseThreshold($){
	my($range) = @_;
	
	# Zerlege Schwellwerte/Bereiche per Regex in Type, Min und Max.
	my($type, $min, $max) = ($range =~ /^(@|)(?:(-?\d+(?:\.\d+)?|~):)?(-?\d+(?:\.\d+)?)?$/);
	
	# Gebe undef zurueck, falls Parsen nicht erfolgreich (durch ungueltiges
	# Format).
	return(undef) unless(defined($type));
	
	# $type ist 0 fuer Alarm falls ausserhalb des Bereichs und 1 fuer Alarm falls
	# innerhalb.
	$type = $type ne '' ? 1 : 0;

	# $min/$max soll undef sein falls keine untere/obere Schwelle definiert.
	$min = defined($min) ? ($min eq '~' ? undef : $min) : 0;
	
	# Pruefe, ob $min > $max ist und vertausche ggf. $min und $max.
	($min, $max) = ($max, $min) if(defined($min) && defined($max) && $min > $max);
	
	return($type, $min, $max);
}

main();
1;

